import { PrismaClient, category, vendor_status, product_status } from '@prisma/client';
import * as bcrypt from 'bcryptjs';

const prisma = new PrismaClient();

async function main() {
    const hashedPassword = await bcrypt.hash('admin123', 10);

    // 0. Activate License if provided via Env (Installer)
    if (process.env.PURCHASE_CODE) {
        console.log('Activating License during seed...');
        await prisma.sitesetting.upsert({
            where: { key: 'license_data' },
            update: {},
            create: {
                key: 'license_data',
                value: JSON.stringify({
                    isValid: true,
                    purchaseCode: process.env.PURCHASE_CODE,
                    type: 'REGULAR',
                    verifiedAt: new Date().toISOString(),
                    lastCheck: new Date().toISOString()
                })
            }
        });
    }

    // 0.1 Seed Default Settings (Fix for mismatching defaults)
    await prisma.sitesetting.upsert({
        where: { key: 'basic_settings' },
        update: {},
        create: {
            key: 'basic_settings',
            value: JSON.stringify({
                site_name: 'DigiCode',
                site_title: 'DigiCode - Premium Digital Assets Marketplace',
                currency: 'USD',
                currency_symbol: '$'
            })
        }
    });

    await prisma.sitesetting.upsert({
        where: { key: 'logo_settings' },
        update: {},
        create: {
            key: 'logo_settings',
            value: JSON.stringify({
                show_site_name: true,
                site_logo: '', // Use default fallback
                favicon: ''
            })
        }
    });

    // 1. Create/Upsert Admin
    const admin = await prisma.user.upsert({
        where: { email: 'admin@digicode.com' },
        update: {},
        create: {
            id: 'user_admin',
            email: 'admin@digicode.com',
            name: 'DigiCode Official',
            passwordHash: hashedPassword,
            role: 'ADMIN',
            storeName: 'Premium Digital Assets',
            storeDescription: 'High-quality software, UI kits, and scripts for developers and designers.',
            vendorStatus: vendor_status.APPROVED,
            updatedAt: new Date(),
        },
    });

    // 2. Expanded Demo Categories
    const categoriesData = [
        { name: 'UI Kits', slug: 'ui-kits' },
        { name: 'PHP Scripts', slug: 'php-scripts' },
        { name: 'WordPress Themes', slug: 'wordpress-themes' },
        { name: 'Mobile Apps', slug: 'mobile-apps' },
        { name: 'Graphics & Vector', slug: 'graphics' },
        { name: 'HTML Templates', slug: 'html-templates' },
        { name: '3D Assets', slug: '3d-assets' },
        { name: 'Plugins', slug: 'plugins' },
    ];

    const seededCategories: category[] = [];
    for (const cat of categoriesData) {
        const createdCat = await prisma.category.upsert({
            where: { slug: cat.slug },
            update: {},
            create: { ...cat, id: cat.slug },
        });
        seededCategories.push(createdCat);
    }

    const getCat = (slug: string): string => {
        const category = seededCategories.find(c => c.slug === slug);
        if (!category) throw new Error(`Category with slug ${slug} not found`);
        return category.id;
    };

    // 3. 15+ Demo Products with matching images
    const productsData = [
        // PHP SCRIPTS
        {
            title: 'DigiCode - Digital Marketplace CMS',
            slug: 'digicode-digital-cms',
            description: '<h1>Ultimate Marketplace Solution</h1><p>A full-featured CMS for selling digital products.</p>',
            price: 59.00,
            categoryId: getCat('php-scripts'),
            vendorId: admin.id,
            coverImage: 'https://images.unsplash.com/photo-1551288049-bebda4e38f71?auto=format&fit=crop&q=80&w=800',
            previewImage: 'https://images.unsplash.com/photo-1551288049-bebda4e38f71?auto=format&fit=crop&q=80&w=1600',
            images: [
                'https://images.unsplash.com/photo-1551288049-bebda4e38f71?auto=format&fit=crop&q=80&w=1600',
                'https://images.unsplash.com/photo-1460925895917-afdab827c52f?auto=format&fit=crop&q=80&w=1600',
                'https://images.unsplash.com/photo-1581291518633-83b4ebd1d83e?auto=format&fit=crop&q=80&w=1600'
            ],
            status: product_status.APPROVED
        },
        {
            title: 'SaaS Pro - Multi-tenant Landing Page',
            slug: 'saas-pro-script',
            description: '<p>A powerful PHP SaaS script with subscription management.</p>',
            price: 45.00,
            categoryId: getCat('php-scripts'),
            vendorId: admin.id,
            coverImage: 'https://images.unsplash.com/photo-1460925895917-afdab827c52f?auto=format&fit=crop&q=80&w=800',
            previewImage: 'https://images.unsplash.com/photo-1460925895917-afdab827c52f?auto=format&fit=crop&q=80&w=1600',
            status: product_status.APPROVED
        },
        // UI KITS
        {
            title: 'CryptoTrade - Wallet UI Kit',
            slug: 'crypto-ui-kit',
            description: '<p>Professional crypto wallet UI kit with 40+ dark themed screens.</p>',
            price: 29.00,
            categoryId: getCat('ui-kits'),
            vendorId: admin.id,
            coverImage: 'https://images.unsplash.com/photo-1639762681485-074b7f938ba0?auto=format&fit=crop&q=80&w=800',
            previewImage: 'https://images.unsplash.com/photo-1639762681485-074b7f938ba0?auto=format&fit=crop&q=80&w=1600',
            status: product_status.APPROVED
        },
        {
            title: 'DashCore - Admin UI Kit',
            slug: 'dashcore-ui',
            description: '<p>Comprehensive dashboard UI kit for modern web apps.</p>',
            price: 34.00,
            categoryId: getCat('ui-kits'),
            vendorId: admin.id,
            coverImage: 'https://images.unsplash.com/photo-1581291518633-83b4ebd1d83e?auto=format&fit=crop&q=80&w=800',
            previewImage: 'https://images.unsplash.com/photo-1581291518633-83b4ebd1d83e?auto=format&fit=crop&q=80&w=1600',
            status: product_status.APPROVED
        },
        // WORDPRESS
        {
            title: 'NewsMag - Professional Blog Theme',
            slug: 'newsmag-wp',
            description: '<p>Speed optimized WordPress theme for newspapers and blogs.</p>',
            price: 49.00,
            categoryId: getCat('wordpress-themes'),
            vendorId: admin.id,
            coverImage: 'https://images.unsplash.com/photo-1499750310107-5fef28a66643?auto=format&fit=crop&q=80&w=800',
            previewImage: 'https://images.unsplash.com/photo-1499750310107-5fef28a66643?auto=format&fit=crop&q=80&w=1600',
            status: product_status.APPROVED
        },
        {
            title: 'EcoStore - WooCommerce Theme',
            slug: 'ecostore-wp',
            description: '<p>Sustainable and eco-friendly shop theme for WordPress.</p>',
            price: 55.00,
            categoryId: getCat('wordpress-themes'),
            vendorId: admin.id,
            coverImage: 'https://images.unsplash.com/photo-1441986300917-64674bd600d8?auto=format&fit=crop&q=80&w=800',
            previewImage: 'https://images.unsplash.com/photo-1441986300917-64674bd600d8?auto=format&fit=crop&q=80&w=1600',
            status: product_status.APPROVED
        },
        // MOBILE APPS
        {
            title: 'Fitness Tracker - Flutter App',
            slug: 'fitness-flutter',
            description: '<p>Complete fitness tracking solution with exercise plans and social sharing.</p>',
            price: 79.00,
            categoryId: getCat('mobile-apps'),
            vendorId: admin.id,
            coverImage: 'https://images.unsplash.com/photo-1510017803434-a899398421b3?auto=format&fit=crop&q=80&w=800',
            previewImage: 'https://images.unsplash.com/photo-1510017803434-a899398421b3?auto=format&fit=crop&q=80&w=1600',
            status: product_status.APPROVED
        },
        {
            title: 'RideShare - Native iOS App',
            slug: 'rideshare-ios',
            description: '<p>Full source code for a modern ride-sharing application.</p>',
            price: 199.00,
            categoryId: getCat('mobile-apps'),
            vendorId: admin.id,
            coverImage: 'https://images.unsplash.com/photo-1555066931-4365d14bab8c?auto=format&fit=crop&q=80&w=800',
            previewImage: 'https://images.unsplash.com/photo-1555066931-4365d14bab8c?auto=format&fit=crop&q=80&w=1600',
            status: product_status.APPROVED
        },
        // HTML TEMPLATES
        {
            title: 'Portfoli - Designer Portfolio',
            slug: 'portfoli-html',
            description: '<p>Creative and minimal HTML portfolio template.</p>',
            price: 15.00,
            categoryId: getCat('html-templates'),
            vendorId: admin.id,
            coverImage: 'https://images.unsplash.com/photo-1507238691740-187a5b1d37b8?auto=format&fit=crop&q=80&w=800',
            previewImage: 'https://images.unsplash.com/photo-1507238691740-187a5b1d37b8?auto=format&fit=crop&q=80&w=1600',
            status: product_status.APPROVED
        },
        {
            title: 'EduHome - Learning Template',
            slug: 'eduhome-html',
            description: '<p>Full-featured HTML template for online courses and education.</p>',
            price: 0.00,
            isFree: true,
            categoryId: getCat('html-templates'),
            vendorId: admin.id,
            coverImage: 'https://images.unsplash.com/photo-1501504905252-473c47e087f8?auto=format&fit=crop&q=80&w=800',
            previewImage: 'https://images.unsplash.com/photo-1501504905252-473c47e087f8?auto=format&fit=crop&q=80&w=1600',
            status: product_status.APPROVED
        },
        // GRAPHICS
        {
            title: 'Modern Logo Pack - 100+ Vectors',
            slug: 'logo-pack-vector',
            description: '<p>A collection of 100+ minimal logo templates in AI and EPS format.</p>',
            price: 19.00,
            categoryId: getCat('graphics'),
            vendorId: admin.id,
            coverImage: 'https://images.unsplash.com/photo-1626785774573-4b799315345d?auto=format&fit=crop&q=80&w=800',
            previewImage: 'https://images.unsplash.com/photo-1626785774573-4b799315345d?auto=format&fit=crop&q=80&w=1600',
            status: product_status.APPROVED
        },
        {
            title: 'Social Media Kit - Instagram Bundle',
            slug: 'social-kit',
            description: '<p>Ready-to-use Instagram posts and stories for business.</p>',
            price: 12.00,
            categoryId: getCat('graphics'),
            vendorId: admin.id,
            coverImage: 'https://images.unsplash.com/photo-1611162617474-5b21e879e113?auto=format&fit=crop&q=80&w=800',
            previewImage: 'https://images.unsplash.com/photo-1611162617474-5b21e879e113?auto=format&fit=crop&q=80&w=1600',
            status: product_status.APPROVED
        },
        // 3D ASSETS
        {
            title: 'Abstract 3D Shapes Pack',
            slug: 'abstract-3d-shapes',
            description: '<p>50+ high-quality 3D icons for your website and apps.</p>',
            price: 25.00,
            categoryId: getCat('3d-assets'),
            vendorId: admin.id,
            coverImage: 'https://images.unsplash.com/photo-1614332287897-cdc485fa562d?auto=format&fit=crop&q=80&w=800',
            previewImage: 'https://images.unsplash.com/photo-1614332287897-cdc485fa562d?auto=format&fit=crop&q=80&w=1600',
            status: product_status.APPROVED
        },
        {
            title: 'Interior Model - Modern Lounge',
            slug: 'lounge-3d-model',
            description: '<p>Realistic 3D interior model for architectural visualization.</p>',
            price: 0.00,
            isFree: true,
            categoryId: getCat('3d-assets'),
            vendorId: admin.id,
            coverImage: 'https://images.unsplash.com/photo-1600210492486-724fe5c67fb0?auto=format&fit=crop&q=80&w=800',
            previewImage: 'https://images.unsplash.com/photo-1600210492486-724fe5c67fb0?auto=format&fit=crop&q=80&w=1600',
            status: product_status.APPROVED
        },
        // PLUGINS
        {
            title: 'Advanced SEO Plugin - Pro',
            slug: 'seo-plugin-pro',
            description: '<p>The most advanced SEO plugin for any web application.</p>',
            price: 39.00,
            categoryId: getCat('plugins'),
            vendorId: admin.id,
            coverImage: 'https://images.unsplash.com/photo-1562577309-4932fdd64cd1?auto=format&fit=crop&q=80&w=800',
            previewImage: 'https://images.unsplash.com/photo-1562577309-4932fdd64cd1?auto=format&fit=crop&q=80&w=1600',
            status: product_status.APPROVED
        }
    ];

    for (const prodRaw of productsData) {
        const prod = prodRaw as any;
        await prisma.product.upsert({
            where: { slug: prod.slug },
            update: {
                ...prod,
                images: prod.images ? JSON.stringify(prod.images) : null,
                updatedAt: new Date()
            },
            create: {
                ...prod,
                id: prod.slug,
                images: prod.images ? JSON.stringify(prod.images) : null,
                updatedAt: new Date()
            },
        });
    }

    // 4. Create Dummy Customers
    const users = [];
    for (let i = 1; i <= 5; i++) {
        const u = await prisma.user.upsert({
            where: { email: `user${i}@demo.com` },
            update: {},
            create: {
                id: `user_${i}`,
                name: `Demo User ${i}`,
                email: `user${i}@demo.com`,
                passwordHash: hashedPassword,
                role: 'CUSTOMER',
                image: `https://i.pravatar.cc/150?u=${i}`,
                updatedAt: new Date()
            }
        });
        users.push(u);
    }

    // 5. Create Dummy Orders & Transactions (For Charts)
    // We'll create random orders for the last 30 days
    console.log('Generating dummy orders...');


    // Simple random ID generator since defaults aren't being picked up
    const genId = () => Math.random().toString(36).substring(2, 15) + Math.random().toString(36).substring(2, 15);

    // Get all product IDs
    const allProducts = await prisma.product.findMany();
    if (allProducts.length > 0) {
        for (let i = 0; i < 20; i++) {
            const randomUser = users[Math.floor(Math.random() * users.length)];
            const randomProduct = allProducts[Math.floor(Math.random() * allProducts.length)];
            const daysAgo = Math.floor(Math.random() * 30);
            const date = new Date();
            date.setDate(date.getDate() - daysAgo);

            const orderId = genId();

            // Create Order
            const order = await prisma.order.create({
                data: {
                    id: orderId,
                    userId: randomUser.id,
                    totalAmount: randomProduct.price,
                    status: 'COMPLETED',
                    createdAt: date,
                    updatedAt: date,
                    orderitem: {
                        create: {
                            id: genId(),
                            productId: randomProduct.id,
                            price: randomProduct.price,
                            vendorEarning: randomProduct.price
                        }
                    }
                }
            });

            // Create Transaction
            await prisma.transaction.create({
                data: {
                    id: genId(),
                    userId: randomUser.id,
                    amount: randomProduct.price,
                    type: 'DEBIT',
                    description: `Order #${order.id}`,
                    createdAt: date
                }
            });
        }
    }


    // 6. Seed Payment Gateways
    console.log('Seeding payment gateways...');
    await prisma.paymentGateway.deleteMany({});

    const gateways = [
        {
            id: 'stripe',
            name: 'Stripe',
            slug: 'stripe',
            type: 'AUTOMATIC',
            isEnabled: true,
            config: JSON.stringify({
                publishableKey: 'pk_test_sample',
                secretKey: 'sk_test_sample'
            })
        },
        {
            id: 'paypal',
            name: 'PayPal',
            slug: 'paypal',
            type: 'AUTOMATIC',
            isEnabled: true,
            config: JSON.stringify({
                clientId: 'sb-client-id',
                clientSecret: 'sb-client-secret',
                mode: 'sandbox' // sandbox or live
            })
        },
        {
            id: 'bank_transfer',
            name: 'Bank Transfer',
            slug: 'manual_bank',
            type: 'MANUAL',
            isEnabled: true,
            config: JSON.stringify({
                bankName: 'Global Bank',
                accountName: 'DigiCode LLC',
                accountNumber: '1234567890',
                swiftCode: 'GBANKUS33',
                instructions: 'Please include your Order ID in the reference.'
            })
        }
    ];

    for (const gw of gateways) {
        try {
            await prisma.paymentGateway.upsert({
                where: { id: gw.id },
                update: {
                    name: gw.name,
                    slug: gw.slug,
                    type: gw.type as any, // Cast if enum mismatch slightly
                    config: gw.config
                },
                create: {
                    id: gw.id,
                    name: gw.name,
                    slug: gw.slug,
                    type: gw.type as any,
                    isEnabled: gw.isEnabled,
                    config: gw.config,
                    createdAt: new Date(),
                    updatedAt: new Date()
                }
            });
        } catch (err: any) {
            console.error('Failed to seed gateway:', gw.id, err.code, err.message);
        }
    }

    console.log('Seed successful:');
    console.log('- Admin:', admin.email);
    console.log('- Categories Total:', seededCategories.length);
    console.log('- Products Total:', productsData.length);
    console.log('- Dummy Users:', users.length);
    console.log('- Dummy Orders Generated');
}

main()
    .catch((e) => {
        console.error(e);
        process.exit(1);
    })
    .finally(async () => {
        await prisma.$disconnect();
    });
